/* $Id: realtime.c,v 1.14 1998/09/16 02:05:15 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

#include <stdlib.h>		/* For exit, strtol */
#include <stdio.h>		/* For printf */
#include <string.h>		/* For strrchr */
#include <unistd.h>		/* For getopt */
#include <sicl.h>		/* For SICL functions */
#include "e1432.h"
#include "err1432.h"

/* Compile flags */
#define	FREQ
#define	SIZE_MIN	1024
#define	SIZE_MAX	2048
#define	NAVG_MIN	1
#define	NAVG_MAX	16
#define	NAVG_STEP	15
#define	DIV_MIN		1
#define	DIV_MAX		4
#define	NCLOCK_FREQ	5
#define	BLOCKS		400
#define	NMOD_MAX	6
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	FIFO_MULT	8

#ifndef	FREQ
#undef	NAVG_MAX
#define	NAVG_MAX	1
#endif

#define	DEBUG(s)	/* s */

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

/* SICL-specific version of the above macro, since SICL error numbers
   can be positive.  Ick. */
#ifdef	__lint
#define	ICHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return -1;\
    }\
} while (func)
#else
#define	ICHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return -1;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: realtime.c,v 1.14 1998/09/16 02:05:15 ericb Exp $";
static const char *progname;

/* Make these global so the interrupt handler can get to them easily */
static INST irq_id[NMOD_MAX];
static E1432ID hw;
static SHORTSIZ16 chan_list[NCHAN_MAX];
static SHORTSIZ16 laddr[NMOD_MAX];
static int group, nchan, nmod, nchan_per_mod, nchan_active;
static int fifo_overflow;
static long expected, block, nblock;

static int
handle_meas_error(long sec)
{
    if ((sec & E1432_IRQ_MEAS_ERROR) != 0)
    {
	DEBUG((void) fprintf(stderr, "%s: *** Fifo overflow detected ***\n",
			     progname));
	fifo_overflow = 1;
    }
    return 0;
}

/*ARGSUSED*/
static int
handle_data(long sec)
{
    FLOATSIZ64 buffer[SIZE_MAX];
    LONGSIZ32 count;
    int     i, mod, scan_count, status;

    /* Quit if we see a fifo overflow */
    if (fifo_overflow)
	return 0;

    /* See if there is data.  Read up to four scans of data */
    scan_count = 0;
    while (scan_count++ < 4)
    {
	CHECK(e1432_print_errors(0));
	status = e1432_block_available(hw, group);
	CHECK(e1432_print_errors(1));

#if 0
	/* Check for mismatch between e1432_block_available and sec */
	if (scan_count == 1)
	{
	    if (status <= 0 && (sec & E1432_IRQ_BLOCK_READY) != 0)
		(void) fprintf(stderr, "%s: block ready irq_handler sec "
			       "and e1432_block_available mismatch\n",
			       progname);
	    if (status == ERR1432_FIFO_OVERRUN && !fifo_overflow)
		(void) fprintf(stderr, "%s: fifo overflow without "
			       "interrupt first\n", progname);
	}
#endif

	/* Print what happened */
	if (status < 0 && status != ERR1432_FIFO_OVERRUN)
	    (void) fprintf(stderr, "%s: e1432_block_available: returned %d\n",
			   progname, status);
	DEBUG(else if (status > 0)
	(void) printf("Block available found!\n"));

	/* If error or no more data, return */
	if (status <= 0)
	    return status;

	/* Read one scan of data */
	for (mod = 0; mod < nmod; mod++)
	    for (i = 0; i < nchan_active; i++)
	    {
		CHECK(e1432_read_float64_data(hw, chan_list[mod *
							   nchan_per_mod + i],
#ifdef	FREQ
					      E1432_FREQ_DATA,
#else
					      E1432_TIME_DATA,
#endif
					      buffer, expected,
					      NULL, &count));
		if (count != expected)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: actual count was %ld, expected %ld\n",
				   progname, count, expected);
		    return -1;
		}
	    }

	/* Increment block count, stop reading data if we're done */
	block++;
	if (block >= nblock)
	    break;
    }
    return 0;
}

static int
handle_reenable(INST id)
{
    int     i;

    /* Don't re-enable interrupts if we're done */
    if (fifo_overflow || block >= nblock)
	return 0;

    /* Find which module did the interrupt */
    for (i = 0; i < nmod; i++)
	if (id == irq_id[i])
	    break;
    if (i == nmod)
	return -1;

    /* Reenable only that module */
    CHECK(e1432_reenable_interrupt(hw, chan_list[nchan_per_mod * i]));

    return 0;
}

/*ARGSUSED*/
static void
irq_handler(INST id, long reason, long sec)
{
#if 0
    int     i;
#endif

    DEBUG((void) printf("irq_handler called\n"));

#if 0
    /* Check for problems */
    for (i = 0; i < nmod; i++)
	if (id == irq_id[i])
	    break;
    if (i == nmod)
	(void) fprintf(stderr, "%s: irq_handler: got wrong id\n",
		       progname);
    if (reason != I_INTR_VXI_SIGNAL)
	(void) fprintf(stderr, "%s: irq_handler: got wrong reason\n",
		       progname);
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != laddr[i])
	(void) fprintf(stderr, "%s: irq_handler: got wrong laddr\n",
		       progname);
#endif

    (void) handle_meas_error(sec);
    (void) handle_data(sec);
    (void) handle_reenable(id);
}

static int
irq_setup(int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	(void) fprintf(stderr, "%s: iopen: returned %d\n", progname, id);
	return -1;
    }
    ICHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    ICHECK(ivxirminfo(id, slot0_laddr, &info));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	(void) fprintf(stderr, "%s: no interrupt lines available\n",
		       progname);
	return -1;
    }
    (void) printf("Using VME interrupt line %d\n", *line);

    /* Set up the interrupt handler routine */
    for (i = 0; i < nmod; i++)
    {
	(void) sprintf(addr, "vxi,%d", laddr[i]);
	irq_id[i] = iopen(addr);
	if (irq_id[i] == 0)
	{
	    (void) fprintf(stderr, "%s: iopen: returned %d\n",
			   progname, irq_id[i]);
	    return -1;
	}
	ICHECK(ionintr(irq_id[i], irq_handler));
	ICHECK(isetintr(irq_id[i], I_INTR_VXI_SIGNAL, 1));
    }

    return 0;
}

static int
init(void)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, status;

    /* Initialize e1432 library, create group of all input channels */
    CHECK(e1432_init_io_driver());
    status = e1432_get_hwconfig(nmod, laddr, hwconfig);
    CHECK(e1432_print_errors(1));
    if (status < 0)
    {
	(void) printf("Installing sema.bin\n");
	CHECK(e1432_install(nmod, laddr, 0, "/opt/e1432/lib/sema.bin"));
	CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));
    }
    CHECK(e1432_assign_channel_numbers(nmod, laddr, &hw));
    nchan = 0;
    for (i = 0; i < nmod; i++)
	nchan += hwconfig[i].input_chans;
    for (i = 0; i < nchan; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);
    nchan_per_mod = nchan / nmod;
    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, group);
	return -1;
    }

    return 0;
}

static int
setup(void)
{
    int     line;

    /* Set up continuous freq data, Hann window, mag-squared */
    CHECK(e1432_set_clock_freq(hw, group, 102400.0));
    CHECK(e1432_set_data_mode(hw, group, E1432_CONTINUOUS_MODE));
#ifdef	FREQ
    CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
    CHECK(e1432_set_enable(hw, group,  E1432_ENABLE_TYPE_TIME,
			   E1432_ENABLE_OFF));
#endif
    CHECK(e1432_set_window(hw, group, E1432_WINDOW_HANNING));
    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_RMS));

    /* Set up interrupts */
    CHECK(irq_setup(&line));
    CHECK(e1432_set_interrupt_priority(hw, group, line));
    CHECK(e1432_set_interrupt_mask(hw, group, E1432_IRQ_MEAS_ERROR));
    CHECK(e1432_set_interrupt_mask(hw, chan_list[nchan - 1],
				   E1432_IRQ_MEAS_ERROR |
				   E1432_IRQ_BLOCK_READY));

    return 0;
}

static int
run(double clock_freq, double span, long blocksize, long navg)
{
    int     i, mod;

    /* Set active channels */
    for (mod = 0; mod < nmod; mod++)
    {
	for (i = 0; i < nchan_active; i++)
	    CHECK(e1432_set_active(hw,
				   chan_list[mod * nchan_per_mod + i],
				   E1432_CHANNEL_ON));
	for (; i < nchan_per_mod; i++)
	    CHECK(e1432_set_active(hw,
				   chan_list[mod * nchan_per_mod + i],
				   E1432_CHANNEL_OFF));
    }

    /* Set other parameters */
    CHECK(e1432_set_clock_freq(hw, group, clock_freq));
    CHECK(e1432_set_span(hw, group, span));
    CHECK(e1432_set_blocksize(hw, group, blocksize));
    CHECK(e1432_set_fifo_size(hw, group, FIFO_MULT * blocksize));
    CHECK(e1432_set_avg_number(hw, group, navg));

    /* variables used by the interrupt routines */
    fifo_overflow = 0;
    block = 0;
    nblock = (BLOCKS + navg - 1) / navg;
    expected = blocksize;
#ifdef	FREQ
    expected /= 2;
#endif

    (void) printf("chans %2d, bs %4ld, navg %4ld, span %.0f -> %.0f kHz: ",
		  nchan_active, blocksize, navg, span, nchan_active *
		  span * nmod);
    (void) fflush(stdout);

    /* Start measurement */
    CHECK(e1432_init_measure(hw, group));

    while (!fifo_overflow && block < nblock)
	(void) sleep(10000);

    if (block < nblock)
    {
	(void) printf("FIFO overflow\n");
	return 0;
    }
    else
    {
	(void) printf("Completed\n");
	return 1;
    }
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [uV] [-n mods]\n"
		   "\t-n: Use <mods> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    double  clock_freq[NCLOCK_FREQ];
    double  span;
    long    blocksize, navg;
    int     i, c, status, divider;
    char   *p;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    nmod = 1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "n:uV")) != -1)
	switch (c)
	{
	case 'n':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 0; i < nmod; i++)
	laddr[i] = 8 + i;

    if (init() < 0)
	return EXIT_FAILURE;
    if (setup() < 0)
	return EXIT_FAILURE;

    /* High clock frequencies which don't allow decimation, and don't
       work with E1432. */
    clock_freq[0] = 196608.0;
    clock_freq[1] = 163840.0;
    clock_freq[2] = 153600.0;
    clock_freq[3] = 133333.3;
    clock_freq[4] = 128000.0;

    /* Run measurements */
    for (i = 0; i < NCLOCK_FREQ; i++)
    {
	span = clock_freq[i] / 2.56;
	for (navg = NAVG_MIN; navg <= NAVG_MAX; navg += NAVG_STEP)
	    for (blocksize = SIZE_MIN;
		 blocksize <= SIZE_MAX;
		 blocksize *= 2)
		for (nchan_active = 1;
		     nchan_active <= nchan_per_mod;
		     nchan_active++)
		{
		    status = run(clock_freq[i], span, blocksize, navg);
		    if (status < 0)
			return EXIT_FAILURE;
		    if (status == 0)
			/* If this many channels failed, presumably so
			   will any larger number of channels */
			break;
		}
    }

    /* Lower clock frequencies which allow decimation and which work
       with E1432, though we would have to decimate at least once with
       E1432. */
    clock_freq[0] = 102400.0;
    clock_freq[1] = 98304.0;
    clock_freq[2] = 81920.0;
    clock_freq[3] = 76800.0;
    clock_freq[4] = 65536.0;

    /* Run measurements */
    for (divider = DIV_MIN; divider <= DIV_MAX; divider *= 2)
	for (i = 0; i < NCLOCK_FREQ; i++)
	{
	    span = clock_freq[i] / 2.56;
	    span /= divider;
	    for (navg = NAVG_MIN; navg <= NAVG_MAX; navg += NAVG_STEP)
		for (blocksize = SIZE_MIN;
		     blocksize <= SIZE_MAX;
		     blocksize *= 2)
		    for (nchan_active = 1;
			 nchan_active <= nchan_per_mod;
			 nchan_active++)
		    {
			status = run(clock_freq[i], span, blocksize, navg);
			if (status < 0)
			    return EXIT_FAILURE;
			if (status == 0)
			    /* If this many channels failed, presumably so
			       will any larger number of channels */
			    break;
		    }
	}

    return EXIT_SUCCESS;
}
